/*
 * Decompiled with CFR 0.152.
 */
package net.osmand.wiki;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.sql.SQLException;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.GZIPOutputStream;
import net.osmand.PlatformUtil;
import org.apache.commons.compress.compressors.bzip2.BZip2CompressorInputStream;
import org.apache.commons.logging.Log;

public abstract class AbstractWikiFilesDownloader {
    private static final Log log = PlatformUtil.getLog(AbstractWikiFilesDownloader.class);
    public static final String DUMPS_WIKIMEDIA_URL = "https://dumps.wikimedia.org/";
    public static final String OTHER_INCR_URL = "other/incr/";
    public static final String LATEST_URL = "/latest/";
    private static final String PAGES_INCR_XML_BZ_2_SUFFIX = "-pages-meta-hist-incr.xml.bz2";
    private static final String LATEST_PAGES_XML_BZ_2_PATTERNS = "-latest-pages-articles\\d+\\.xml.+bz2\"";
    private final List<String> downloadedPageFiles = new ArrayList<String>();
    private long maxId = 0L;
    public final String USER_AGENT = "OsmAnd-Bot/1.0 (+https://osmand.net; support@osmand.net) OsmAndJavaServer/1.0";
    private static final String TIMESTAMP_FILE = "timestamp.txt";
    private static final int INITIAL_WAIT_TIME_MS = 1000;
    private static final int MAX_RETRIES = 5;
    private static final int CONNECT_TIMEOUT = 30000;
    private static final int READ_TIMEOUT = 60000;

    public AbstractWikiFilesDownloader(File wikiDB, boolean daily, DownloadHandler dh) {
        try {
            List<FileForDBUpdate> updateFileList;
            log.info((Object)"Start %s download".formatted(this.getFilePrefix()));
            this.maxId = daily ? 0L : this.getMaxIdFromDb(wikiDB);
            String destFolder = wikiDB.getParent();
            if (daily) {
                Instant instant = this.readTimestampFile(destFolder);
                LocalDate lastModifiedDate = instant.atZone(ZoneId.systemDefault()).toLocalDate();
                updateFileList = this.getLatestFilesURL(this.getWikiIncrDirURL(), lastModifiedDate);
            } else {
                updateFileList = this.readFilesUrl(this.getWikiLatestDirURL());
            }
            this.downloadPageFiles(destFolder, updateFileList, dh);
            if (daily) {
                this.writeTimestampFile(destFolder);
            }
            log.info((Object)"Finish %s download".formatted(this.getFilePrefix()));
        }
        catch (IOException | SQLException e) {
            e.printStackTrace();
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
    }

    public abstract String getFilePrefix();

    public abstract long getMaxPageId() throws IOException;

    public abstract long getMaxIdFromDb(File var1) throws SQLException;

    public String getWikiLatestDirURL() {
        return DUMPS_WIKIMEDIA_URL + this.getFilePrefix() + LATEST_URL;
    }

    public String getWikiIncrDirURL() {
        return "https://dumps.wikimedia.org/other/incr/" + this.getFilePrefix() + "/";
    }

    private String getLatestFilesPattern() {
        return this.getFilePrefix() + LATEST_PAGES_XML_BZ_2_PATTERNS;
    }

    public long getMaxQId() {
        return this.maxId;
    }

    public List<String> getDownloadedPageFiles() {
        return this.downloadedPageFiles;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    Instant readTimestampFile(String pathToTimestamp) {
        Path timestampFile = Path.of(pathToTimestamp, TIMESTAMP_FILE);
        if (!Files.exists(timestampFile, new LinkOption[0])) {
            return Instant.EPOCH;
        }
        try (BufferedReader reader = Files.newBufferedReader(timestampFile);){
            String line = reader.readLine();
            if (line == null || line.isBlank()) {
                Instant instant2 = Instant.EPOCH;
                return instant2;
            }
            Instant instant = Instant.parse(line.trim());
            return instant;
        }
        catch (IOException | DateTimeParseException e) {
            return Instant.EPOCH;
        }
    }

    void writeTimestampFile(String pathToTimestamp) {
        Instant now = Instant.now();
        try (FileWriter writer = new FileWriter(new File(pathToTimestamp, TIMESTAMP_FILE));){
            writer.write(now.toString());
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private void downloadPageFiles(String destFolder, List<FileForDBUpdate> updateList, DownloadHandler dh) throws IOException {
        for (FileForDBUpdate p : updateList) {
            String fileName = p.url.substring(p.url.lastIndexOf("/") + 1);
            Object gzFile = fileName.replace(".bz2", ".gz");
            File gz = new File((String)(gzFile = destFolder + "/" + (String)gzFile));
            if (gz.exists()) {
                System.out.println((String)gzFile + " already downloaded");
                this.downloadedPageFiles.add((String)gzFile);
                if (dh == null) continue;
                dh.onFinishDownload((String)gzFile);
                continue;
            }
            try (BufferedInputStream in = new BufferedInputStream(new URL(p.url).openStream());
                 BZip2CompressorInputStream bzIn = new BZip2CompressorInputStream((InputStream)in);
                 GZIPOutputStream gzOut = new GZIPOutputStream(new FileOutputStream((String)gzFile));){
                int n;
                byte[] buffer = new byte[8192];
                while ((n = bzIn.read(buffer)) != -1) {
                    gzOut.write(buffer, 0, n);
                }
            }
            System.out.println((String)gzFile + " downloading is finished");
            this.downloadedPageFiles.add((String)gzFile);
            if (dh == null) continue;
            dh.onFinishDownload((String)gzFile);
        }
    }

    public void removeDownloadedPages() {
        for (String s : this.downloadedPageFiles) {
            File f = new File(s);
            if (!f.exists()) continue;
            f.delete();
        }
    }

    private List<FileForDBUpdate> readFilesUrl(String wikiUrl) throws IOException, InterruptedException {
        String line;
        Pattern pattern = Pattern.compile(this.getLatestFilesPattern());
        HttpURLConnection connection = this.openConnectionWithRetry(wikiUrl);
        long maxPageId = this.getMaxPageId();
        InputStream inputStream = connection.getInputStream();
        BufferedReader r = new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.UTF_8));
        ArrayList<FileForDBUpdate> result = new ArrayList<FileForDBUpdate>();
        while ((line = r.readLine()) != null) {
            Matcher matcher = pattern.matcher(line);
            if (!matcher.find()) continue;
            String data = line.replace("<a href=\"", "").replaceAll("\">.+$", "");
            String p = data.replaceAll(".+xml-", "").replaceAll("\\.bz2", "");
            String[] pages = p.split("p");
            int min = Integer.parseInt(pages[1]);
            int max = Integer.parseInt(pages[2]);
            FileForDBUpdate fileForUpdate = new FileForDBUpdate();
            fileForUpdate.url = wikiUrl + data;
            if (maxPageId >= (long)max && maxPageId >= (long)min) continue;
            result.add(fileForUpdate);
        }
        connection.disconnect();
        return result;
    }

    private List<FileForDBUpdate> getLatestFilesURL(String wikiUrl, LocalDate lastUpdateDate) throws IOException, InterruptedException {
        String line;
        Pattern pattern = Pattern.compile("<a href=\"\\d{8}/\">\\d{8}/</a>");
        HttpURLConnection connection = this.openConnectionWithRetry(wikiUrl);
        InputStream inputStream = connection.getInputStream();
        BufferedReader r = new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.UTF_8));
        ArrayList<FileForDBUpdate> result = new ArrayList<FileForDBUpdate>();
        while ((line = r.readLine()) != null) {
            FileForDBUpdate fileForUpdate;
            Matcher matcher = pattern.matcher(line);
            if (!matcher.find() || (fileForUpdate = this.getIncrFile(wikiUrl, line, lastUpdateDate)) == null) continue;
            result.add(fileForUpdate);
        }
        connection.disconnect();
        return result;
    }

    private FileForDBUpdate getIncrFile(String wikiUrl, String line, LocalDate lastUpdateDate) {
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyyMMdd");
        String fileDate = line.replace("<a href=\"", "").replaceAll("/\">.+$", "");
        LocalDate date = LocalDate.parse(fileDate, formatter);
        FileForDBUpdate fileForUpdate = null;
        if (date.isAfter(lastUpdateDate) || date.isEqual(lastUpdateDate)) {
            fileForUpdate = new FileForDBUpdate();
            fileForUpdate.url = wikiUrl + fileDate + "/" + this.getFilePrefix() + "-" + fileDate + PAGES_INCR_XML_BZ_2_SUFFIX;
        }
        return fileForUpdate;
    }

    public HttpURLConnection openConnectionWithRetry(String urlString) throws IOException, InterruptedException {
        int attempt = 0;
        long waitTime = 1000L;
        while (true) {
            ++attempt;
            HttpURLConnection connection = (HttpURLConnection)new URL(urlString).openConnection();
            connection.setRequestProperty("User-Agent", "OsmAnd-Bot/1.0 (+https://osmand.net; support@osmand.net) OsmAndJavaServer/1.0");
            connection.setConnectTimeout(30000);
            connection.setReadTimeout(60000);
            try {
                int code = connection.getResponseCode();
                if (code == 200) {
                    return connection;
                }
                if (code == 429) {
                    String retryAfter = connection.getHeaderField("Retry-After");
                    long waitSeconds = retryAfter != null ? Long.parseLong(retryAfter) : waitTime / 1000L;
                    System.err.printf("429 Too Many Requests (attempt %d). Waiting %ds...%n", attempt, waitSeconds);
                    Thread.sleep(waitSeconds * 1000L);
                    waitTime *= 2L;
                    continue;
                }
                if (code >= 500 && code < 600) {
                    System.err.printf("Server error %d (attempt %d). Retrying in %d ms...%n", code, attempt, waitTime);
                    Thread.sleep(waitTime);
                    waitTime *= 2L;
                    continue;
                }
                throw new IOException("HTTP error " + code + " for URL " + urlString);
            }
            catch (IOException e) {
                connection.disconnect();
                if (attempt >= 5) {
                    throw new IOException("Failed after retries: " + e.getMessage(), e);
                }
                System.err.printf("Connection error (attempt %d): %s%n", attempt, e.getMessage());
                Thread.sleep(waitTime);
                waitTime *= 2L;
                continue;
            }
            break;
        }
    }

    public static interface DownloadHandler {
        public void onFinishDownload(String var1);
    }

    private static class FileForDBUpdate {
        String url;

        private FileForDBUpdate() {
        }
    }
}

