/*
 * Decompiled with CFR 0.152.
 */
package net.osmand.util;

import com.google.protobuf.CodedOutputStream;
import com.jcraft.jsch.Channel;
import com.jcraft.jsch.ChannelExec;
import com.jcraft.jsch.JSch;
import com.jcraft.jsch.JSchException;
import com.jcraft.jsch.Session;
import gnu.trove.list.array.TIntArrayList;
import gnu.trove.map.TIntObjectMap;
import gnu.trove.map.hash.TLongObjectHashMap;
import java.io.BufferedReader;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.Enumeration;
import java.util.List;
import java.util.Locale;
import java.util.Properties;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;
import net.osmand.PlatformUtil;
import net.osmand.ResultMatcher;
import net.osmand.binary.BinaryIndexPart;
import net.osmand.binary.BinaryMapDataObject;
import net.osmand.binary.BinaryMapIndexReader;
import net.osmand.obf.BinaryMerger;
import net.osmand.obf.preparation.AbstractIndexPartCreator;
import net.osmand.obf.preparation.BinaryFileReference;
import net.osmand.obf.preparation.BinaryMapIndexWriter;
import net.osmand.obf.preparation.IndexVectorMapCreator;
import net.osmand.util.Algorithms;
import net.osmand.util.MapUtils;
import org.apache.commons.logging.Log;
import rtree.LeafElement;
import rtree.Node;
import rtree.RTree;
import rtree.RTreeException;
import rtree.Rect;

public class IndexUploader {
    protected static final Log log = PlatformUtil.getLog(IndexUploader.class);
    private static final double MIN_SIZE_TO_UPLOAD = 0.001;
    private static final int BUFFER_SIZE = 32768;
    private static final int MB = 0x100000;
    private File directory;
    private File targetDirectory;
    private UploadCredentials uploadCredentials = new DummyCredentials();
    private FileFilter fileFilter = new FileFilter();
    private FileFilter deleteFileFilter = null;
    private boolean roadProcess;
    private boolean wikiProcess;
    private boolean srtmProcess;
    private boolean travelProcess;
    private boolean depthProcess;
    private boolean mapsProcess;
    private int numberOfThreads = 2;

    public IndexUploader(String path, String targetPath) throws IndexUploadException {
        this.directory = new File(path);
        if (!this.directory.isDirectory()) {
            throw new IndexUploadException("Not a directory:" + path);
        }
        this.targetDirectory = new File(targetPath);
        if (!this.targetDirectory.isDirectory()) {
            throw new IndexUploadException("Not a directory:" + targetPath);
        }
    }

    public static void main(String[] args) throws IOException, IndexUploadException, RTreeException {
        try {
            String srcPath;
            String targetPath = srcPath = IndexUploader.extractDirectory(args, 0);
            if (args.length > 1) {
                targetPath = IndexUploader.extractDirectory(args, 1);
            }
            IndexUploader indexUploader = new IndexUploader(srcPath, targetPath);
            if (args.length > 2) {
                indexUploader.parseParameters(args, 2);
            }
            indexUploader.run();
        }
        catch (IndexUploadException e) {
            log.error((Object)e.getMessage());
        }
    }

    private void parseParameters(String[] args, int start) throws IndexUploadException {
        int idx = this.parseFilter(args, start);
        if (args.length > idx) {
            this.parseUploadCredentials(args, idx);
        }
    }

    private int parseFilter(String[] args, int start) throws IndexUploadException {
        int p;
        FileFilter fileFilter = null;
        do {
            p = start;
            if (args[start].startsWith("--ff=")) {
                try {
                    if (fileFilter == null) {
                        fileFilter = new FileFilter();
                    }
                    fileFilter.parseFile(args[start].substring("--ff=".length()));
                }
                catch (IOException e) {
                    throw new IndexUploadException(e.getMessage());
                }
                ++start;
                continue;
            }
            if (args[start].startsWith("--fp=")) {
                if (fileFilter == null) {
                    fileFilter = new FileFilter();
                }
                fileFilter.parseCSV(args[start].substring("--fp=".length()));
                ++start;
                continue;
            }
            if (args[start].startsWith("--ep=")) {
                if (fileFilter == null) {
                    throw new NullPointerException();
                }
                fileFilter.parseExcludeCSV(args[start].substring("--ep=".length()));
                ++start;
                continue;
            }
            if (args[start].startsWith("--dp=")) {
                this.deleteFileFilter = new FileFilter();
                this.deleteFileFilter.parseCSV(args[start].substring("--dp=".length()));
                ++start;
                continue;
            }
            if (args[start].startsWith("--roads")) {
                this.roadProcess = true;
                ++start;
                continue;
            }
            if (args[start].startsWith("--nt=")) {
                this.numberOfThreads = Integer.parseInt(args[start].substring("--nt=".length()));
                ++start;
                continue;
            }
            if (args[start].startsWith("--wiki")) {
                this.wikiProcess = true;
                ++start;
                continue;
            }
            if (args[start].startsWith("--srtm")) {
                this.srtmProcess = true;
                ++start;
                continue;
            }
            if (args[start].startsWith("--travel")) {
                this.travelProcess = true;
                ++start;
                continue;
            }
            if (args[start].startsWith("--depth")) {
                this.depthProcess = true;
                ++start;
                continue;
            }
            if (!args[start].startsWith("--maps")) continue;
            this.mapsProcess = true;
            ++start;
        } while (p != start);
        if (fileFilter != null) {
            this.fileFilter = fileFilter;
        }
        return start;
    }

    protected void parseUploadCredentials(String[] args, int start) throws IndexUploadException {
        if (!"-ssh".equals(args[start])) {
            return;
        }
        this.uploadCredentials = new UploadSSHCredentials();
        for (int i = start + 1; i < args.length; ++i) {
            this.uploadCredentials.parseParameter(args[i]);
        }
    }

    public void setUploadCredentials(UploadCredentials uploadCredentials) {
        this.uploadCredentials = uploadCredentials;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void run() throws IndexUploadException, IOException, RTreeException {
        try {
            this.uploadCredentials.connect();
            File[] listFiles = this.directory.listFiles();
            ExecutorService service = Executors.newFixedThreadPool(this.numberOfThreads);
            for (final File f : listFiles) {
                if (!this.checkFileNeedToBeUploaded(f)) continue;
                service.submit(new Runnable(){

                    @Override
                    public void run() {
                        IndexUploader.this.processFile(f);
                    }
                });
            }
            log.error((Object)"Stopping the tasks queue...");
            service.shutdown();
            log.error((Object)"Waiting termination all tasks...");
            service.awaitTermination(24L, TimeUnit.HOURS);
            if (this.deleteFileFilter != null) {
                log.error((Object)("Delete file filter is not supported with this credentions (method) " + this.uploadCredentials));
            }
            log.error((Object)"All tasks has finished.");
        }
        catch (InterruptedException e) {
            log.error((Object)("Await failed: " + e.getMessage()), (Throwable)e);
        }
        finally {
            this.uploadCredentials.disconnect();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void processFile(File f) {
        try {
            File unzippedFolder;
            String fileName = f.getName();
            File mainFile = unzippedFolder = this.unzip(f);
            if (unzippedFolder.isDirectory()) {
                for (File tf : unzippedFolder.listFiles()) {
                    if (!tf.getName().endsWith(".obf")) continue;
                    mainFile = tf;
                    break;
                }
            }
            boolean skip = false;
            try {
                if (this.mapsProcess) {
                    boolean worldFile;
                    boolean bl = worldFile = fileName.toLowerCase().contains("basemap") || fileName.toLowerCase().contains("world");
                    if (!worldFile && !fileName.contains("_ext_")) {
                        IndexUploader.extractRoadOnlyFile(mainFile, new File(this.directory, fileName.replace(".obf", ".road.obf")));
                    }
                }
                String description = this.checkfileAndGetDescription(mainFile);
                long timestampCreated = mainFile.lastModified();
                if (description == null) {
                    log.info((Object)("Skip file empty description " + f.getName()));
                    skip = true;
                } else {
                    File zFile = new File(f.getParentFile(), unzippedFolder.getName() + ".zip");
                    IndexUploader.zip(unzippedFolder, zFile, description, timestampCreated);
                    this.uploadIndex(f, zFile, description, this.uploadCredentials);
                }
            }
            finally {
                if (!(skip || f.getName().equals(unzippedFolder.getName()) && (this.targetDirectory == null || this.targetDirectory.equals(this.directory)))) {
                    Algorithms.removeAllFiles((File)unzippedFolder);
                }
            }
        }
        catch (IOException | RuntimeException | OneFileException | RTreeException e) {
            log.error((Object)(f.getName() + ": " + e.getMessage()), (Throwable)e);
        }
    }

    private boolean checkFileNeedToBeUploaded(File f) {
        if (!this.fileFilter.fileCanBeUploaded(f)) {
            return false;
        }
        long timestampCreated = f.lastModified();
        if (!this.uploadCredentials.checkIfUploadNeededByTimestamp(f.getName(), timestampCreated)) {
            log.info((Object)("File skipped because timestamp was not changed " + f.getName()));
            return false;
        }
        String fileName = f.getName();
        log.info((Object)("Process file " + f.getName()));
        boolean skip = false;
        if ((fileName.contains(".srtm") || fileName.contains(".srtmf")) != this.srtmProcess) {
            skip = true;
        } else if (fileName.contains(".road") != this.roadProcess) {
            skip = true;
        } else if (fileName.contains(".wiki") != this.wikiProcess) {
            skip = true;
        } else if (fileName.contains(".travel") != this.travelProcess) {
            skip = true;
        } else if (fileName.contains(".depth") != this.depthProcess) {
            skip = true;
        }
        if (skip) {
            log.info((Object)("Skip file: " + f.getName()));
            return false;
        }
        return true;
    }

    public static File zip(File folder, File zFile, String description, long lastModifiedTime) throws OneFileException {
        try {
            ZipOutputStream zout = new ZipOutputStream(new FileOutputStream(zFile));
            zout.setLevel(9);
            Collection<File> lfs = folder.isFile() ? Collections.singleton(folder) : Arrays.asList(folder.listFiles());
            for (File f : lfs) {
                log.info((Object)("Zipping to file:" + zFile.getName() + " with desc:" + description));
                IndexUploader.putZipEntry(description, "", lastModifiedTime, zout, f);
            }
            Algorithms.closeStream((Closeable)zout);
            zFile.setLastModified(lastModifiedTime);
        }
        catch (IOException e) {
            throw new OneFileException("cannot zip file:" + e.getMessage());
        }
        return zFile;
    }

    private static void putZipEntry(String description, String parentEntry, long lastModifiedTime, ZipOutputStream zout, File f) throws IOException, FileNotFoundException {
        if (f.isDirectory()) {
            for (File lf : f.listFiles()) {
                IndexUploader.putZipEntry(description, parentEntry + f.getName() + "/", lastModifiedTime, zout, lf);
            }
        } else {
            log.info((Object)("Zipping file:" + f.getName() + " with desc:" + description));
            ZipEntry zEntry = new ZipEntry(parentEntry + f.getName());
            zEntry.setSize(f.length());
            zEntry.setComment(description);
            zEntry.setTime(lastModifiedTime);
            zout.putNextEntry(zEntry);
            FileInputStream is = new FileInputStream(f);
            Algorithms.streamCopy((InputStream)is, (OutputStream)zout);
            Algorithms.closeStream((Closeable)is);
        }
    }

    private String checkfileAndGetDescription(File mainFile) throws OneFileException, IOException, RTreeException {
        String fileName = mainFile.getName();
        if (fileName.endsWith(".obf")) {
            RandomAccessFile raf = null;
            try {
                if (mainFile.length() > Integer.MAX_VALUE) {
                    throw new OneFileException(String.format("File size exceeds maximum supported - %s %d MB", mainFile.getName(), mainFile.length() >> 20));
                }
                raf = new RandomAccessFile(mainFile, "r");
                BinaryMapIndexReader reader = new BinaryMapIndexReader(raf, mainFile);
                if (reader.getVersion() != 2) {
                    throw new OneFileException("Uploader version is not compatible " + reader.getVersion() + " to current 2");
                }
                String summary = this.getDescription(reader, fileName);
                reader.close();
                boolean setLastModified = mainFile.setLastModified(reader.getDateCreated());
                if (!setLastModified) {
                    File copy = new File(mainFile.getAbsolutePath() + ".cp");
                    Algorithms.fileCopy((File)mainFile, (File)copy);
                    mainFile.delete();
                    copy.renameTo(mainFile);
                    setLastModified = mainFile.setLastModified(reader.getDateCreated());
                }
                return summary;
            }
            catch (IOException e) {
                if (raf != null) {
                    try {
                        raf.close();
                    }
                    catch (IOException iOException) {
                        // empty catch block
                    }
                }
                throw new OneFileException("Reader could not read the index: " + e.getMessage());
            }
        }
        throw new OneFileException("Not supported file format " + fileName);
    }

    public static synchronized void extractRoadOnlyFile(File mainFile, File roadOnlyFile) throws IOException, RTreeException {
        RandomAccessFile raf = new RandomAccessFile(mainFile, "r");
        BinaryMapIndexReader index = new BinaryMapIndexReader(raf, mainFile);
        final RandomAccessFile routf = new RandomAccessFile(roadOnlyFile, "rw");
        routf.setLength(0L);
        CodedOutputStream ous = CodedOutputStream.newInstance((OutputStream)new OutputStream(){

            @Override
            public void write(int b) throws IOException {
                routf.write(b);
            }

            @Override
            public void write(byte[] b) throws IOException {
                routf.write(b);
            }

            @Override
            public void write(byte[] b, int off, int len) throws IOException {
                routf.write(b, off, len);
            }
        });
        byte[] BUFFER_TO_READ = new byte[32768];
        ous.writeInt32(1, index.getVersion());
        ous.writeInt64(18, index.getDateCreated());
        for (int i = 0; i < index.getIndexes().size(); ++i) {
            BinaryIndexPart part = (BinaryIndexPart)index.getIndexes().get(i);
            if (part instanceof BinaryMapIndexReader.MapIndex) {
                IndexUploader.copyMapIndex(roadOnlyFile, (BinaryMapIndexReader.MapIndex)part, index, ous, raf, routf);
                continue;
            }
            ous.writeTag(part.getFieldNumber(), 6);
            BinaryMerger.writeInt(ous, part.getLength());
            BinaryMerger.copyBinaryPart(ous, BUFFER_TO_READ, raf, part.getFilePointer(), part.getLength());
        }
        ous.writeInt32(32, index.getVersion());
        ous.flush();
        routf.close();
        raf.close();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void copyMapIndex(File roadOnlyFile, BinaryMapIndexReader.MapIndex part, BinaryMapIndexReader index, CodedOutputStream ous, RandomAccessFile raf, RandomAccessFile routf) throws IOException, RTreeException {
        List rts = part.getRoots();
        BinaryMapIndexWriter writer = new BinaryMapIndexWriter(routf, ous);
        writer.startWriteMapIndex(part.getName());
        boolean first = true;
        for (BinaryMapIndexReader.MapRoot r : rts) {
            if (r.getMaxZoom() <= 10) {
                if (first) {
                    throw new UnsupportedOperationException("Can't write top level zoom");
                }
                ous.writeTag(5, 6);
                BinaryMerger.writeInt(ous, r.getLength());
                BinaryMerger.copyBinaryPart(ous, new byte[32768], raf, r.getFilePointer(), r.getLength());
                continue;
            }
            TLongObjectHashMap objects = new TLongObjectHashMap();
            File nonpackRtree = new File(roadOnlyFile.getParentFile(), "nonpack" + r.getMinZoom() + "." + roadOnlyFile.getName() + ".rtree");
            File packRtree = new File(roadOnlyFile.getParentFile(), "pack" + r.getMinZoom() + "." + roadOnlyFile.getName() + ".rtree");
            RTree rtree = null;
            try {
                rtree = new RTree(nonpackRtree.getAbsolutePath());
                BinaryMapIndexReader.SearchRequest<BinaryMapDataObject> req = IndexUploader.buildSearchRequest(r, (TLongObjectHashMap<BinaryMapDataObject>)objects, rtree);
                index.searchMapIndex(req, part);
                if (first) {
                    first = false;
                    writer.writeMapEncodingRules((TIntObjectMap<BinaryMapIndexReader.TagValuePair>)part.decodingRules);
                }
                rtree = AbstractIndexPartCreator.packRtreeFile(rtree, nonpackRtree.getAbsolutePath(), packRtree.getAbsolutePath());
                TLongObjectHashMap treeHeader = new TLongObjectHashMap();
                long rootIndex = rtree.getFileHdr().getRootIndex();
                Node root = rtree.getReadNode(rootIndex);
                Rect rootBounds = IndexVectorMapCreator.calcBounds(root);
                if (rootBounds == null) continue;
                writer.startWriteMapLevelIndex(r.getMinZoom(), r.getMaxZoom(), rootBounds.getMinX(), rootBounds.getMaxX(), rootBounds.getMinY(), rootBounds.getMaxY());
                IndexVectorMapCreator.writeBinaryMapTree(root, rootBounds, rtree, writer, (TLongObjectHashMap<BinaryFileReference>)treeHeader);
                IndexVectorMapCreator.writeBinaryMapBlock(root, rootBounds, rtree, writer, (TLongObjectHashMap<BinaryFileReference>)treeHeader, (TLongObjectHashMap<BinaryMapDataObject>)objects, r.getMapZoom(), false);
                writer.endWriteMapLevelIndex();
            }
            finally {
                if (rtree != null) {
                    RandomAccessFile file = rtree.getFileHdr().getFile();
                    file.close();
                }
                nonpackRtree.delete();
                packRtree.delete();
                RTree.clearCache();
            }
        }
        writer.endWriteMapIndex();
    }

    private static double polygonArea(BinaryMapDataObject obj) {
        int zoom = 16;
        float mult = (float)(1.0 / MapUtils.getPowZoom((double)Math.max(31 - (zoom + 8), 0)));
        double area = 0.0;
        int j = obj.getPointsLength() - 1;
        int i = 0;
        while (i < obj.getPointsLength()) {
            int px = obj.getPoint31XTile(i);
            int py = obj.getPoint31YTile(i);
            int sx = obj.getPoint31XTile(j);
            int sy = obj.getPoint31YTile(j);
            area += (double)(((float)sx + (float)px) * ((float)sy - (float)py));
            j = i++;
        }
        return Math.abs(area) * (double)mult * (double)mult * 0.5;
    }

    private static BinaryMapIndexReader.SearchRequest<BinaryMapDataObject> buildSearchRequest(BinaryMapIndexReader.MapRoot r, final TLongObjectHashMap<BinaryMapDataObject> objects, final RTree urTree) {
        BinaryMapIndexReader.SearchRequest req = BinaryMapIndexReader.buildSearchRequest((int)0, (int)Integer.MAX_VALUE, (int)0, (int)Integer.MAX_VALUE, (int)r.getMinZoom(), (BinaryMapIndexReader.SearchFilter)new BinaryMapIndexReader.SearchFilter(){

            public boolean accept(TIntArrayList types, BinaryMapIndexReader.MapIndex index) {
                return true;
            }
        }, (ResultMatcher)new ResultMatcher<BinaryMapDataObject>(){

            public boolean publish(BinaryMapDataObject obj) {
                int minX = obj.getPoint31XTile(0);
                int maxX = obj.getPoint31XTile(0);
                int maxY = obj.getPoint31YTile(0);
                int minY = obj.getPoint31YTile(0);
                for (int i = 1; i < obj.getPointsLength(); ++i) {
                    minX = Math.min(minX, obj.getPoint31XTile(i));
                    minY = Math.min(minY, obj.getPoint31YTile(i));
                    maxX = Math.max(maxX, obj.getPoint31XTile(i));
                    maxY = Math.max(maxY, obj.getPoint31YTile(i));
                }
                if (IndexUploader.accept(obj, minX, maxX, minY, maxY)) {
                    objects.put(obj.getId(), (Object)obj);
                    try {
                        urTree.insert(new LeafElement(new Rect(minX, minY, maxX, maxY), obj.getId()));
                    }
                    catch (Exception e) {
                        throw new RuntimeException(e);
                    }
                }
                return false;
            }

            public boolean isCancelled() {
                return false;
            }
        });
        return req;
    }

    protected static boolean accept(BinaryMapDataObject obj, int minX, int maxX, int minY, int maxY) {
        if (obj.isArea() && IndexUploader.polygonArea(obj) > 1000.0) {
            return true;
        }
        boolean point = obj.getPointsLength() == 1;
        for (int i = 0; i < obj.getTypes().length; ++i) {
            BinaryMapIndexReader.TagValuePair tv = obj.getMapIndex().decodeType(obj.getTypes()[i]);
            if (tv.tag.equals("waterway")) {
                return true;
            }
            if (tv.tag.equals("place") && point) {
                return true;
            }
            if (tv.tag.equals("highway") && point) {
                return true;
            }
            if (tv.tag.equals("crossing") && point) {
                return true;
            }
            if (!tv.tag.equals("public_transport") || !point) continue;
            return true;
        }
        return false;
    }

    private String getDescription(BinaryMapIndexReader reader, String fileName) {
        Object summary = " data for ";
        boolean fir = true;
        if (fileName.contains(".srtm")) {
            summary = "Contour lines " + (fileName.contains(".srtmf") ? "feet " : "meters ");
        } else if (fileName.contains(".travel")) {
            summary = "Travel guide" + (String)summary;
        } else if (fileName.contains(".depth")) {
            summary = "Depth" + (String)summary;
        } else if (fileName.contains("_wiki.")) {
            summary = "Wikipedia" + (String)summary;
        } else {
            if (reader.containsAddressData()) {
                summary = "Address" + (fir ? "" : ", ") + (String)summary;
                fir = false;
            }
            if (reader.hasTransportData()) {
                summary = "Transport" + (fir ? "" : ", ") + (String)summary;
                fir = false;
            }
            if (reader.containsPoiData()) {
                summary = "POI" + (fir ? "" : ", ") + (String)summary;
                fir = false;
            }
            if (reader.containsRouteData()) {
                summary = "Roads" + (fir ? "" : ", ") + (String)summary;
                fir = false;
            }
            if (reader.containsMapData()) {
                summary = "Map" + (fir ? "" : ", ") + (String)summary;
                fir = false;
            }
        }
        int last = fileName.lastIndexOf(95, fileName.indexOf(46));
        if (last == -1) {
            last = fileName.indexOf(46);
        }
        String regionName = fileName.substring(0, last);
        summary = (String)summary + regionName;
        summary = ((String)summary).replace('_', ' ');
        return summary;
    }

    private File unzip(File f) throws OneFileException {
        ZipFile zipFile = null;
        try {
            if (!Algorithms.isZipFile((File)f)) {
                File file = f;
                return file;
            }
            log.info((Object)("Unzipping file: " + f.getName()));
            zipFile = new ZipFile(f);
            Enumeration<? extends ZipEntry> entries = zipFile.entries();
            long slastModified = f.lastModified();
            String folderName = f.getName().substring(0, f.getName().length() - 4);
            File unzipFolder = new File(f.getParentFile(), folderName);
            unzipFolder.mkdirs();
            while (entries.hasMoreElements()) {
                ZipEntry entry = entries.nextElement();
                long lastModified = slastModified;
                if (entry.isDirectory()) continue;
                if (entry.getTime() < lastModified) {
                    lastModified = entry.getTime();
                }
                File tempFile = new File(unzipFolder, entry.getName());
                tempFile.getParentFile().mkdirs();
                InputStream zin = zipFile.getInputStream(entry);
                FileOutputStream out = new FileOutputStream(tempFile);
                Algorithms.streamCopy((InputStream)zin, (OutputStream)out);
                Algorithms.closeStream((Closeable)zin);
                Algorithms.closeStream((Closeable)out);
                tempFile.setLastModified(lastModified);
            }
            File file = unzipFolder;
            return file;
        }
        catch (ZipException e) {
            throw new OneFileException("cannot unzip:" + e.getMessage(), e);
        }
        catch (IOException e) {
            throw new OneFileException("cannot unzip:" + e.getMessage(), e);
        }
        finally {
            if (zipFile != null) {
                try {
                    zipFile.close();
                }
                catch (IOException e) {
                    throw new OneFileException("cannot unzip:" + e.getMessage(), e);
                }
            }
        }
    }

    private void uploadIndex(File srcFile, File zipFile, String summary, UploadCredentials uc) {
        double mbLengh = (double)zipFile.length() / 1048576.0;
        File toUpload = zipFile;
        String fileName = toUpload.getName();
        String logFileName = fileName.substring(0, fileName.lastIndexOf(46)) + ".gen.log";
        if (mbLengh < 0.001) {
            log.info((Object)("Skip uploading index due to size " + fileName));
        }
        try {
            log.info((Object)("Uploading index " + fileName + " (log file: " + logFileName + ")"));
            boolean uploaded = this.uploadFileToServer(toUpload, summary, uc);
            if (!uploaded) {
                log.info((Object)"Upload failed");
            }
            if (uploaded && this.targetDirectory != null && !this.targetDirectory.equals(this.directory)) {
                File toBackup = new File(this.targetDirectory, fileName);
                File logFile = new File(zipFile.getParentFile(), logFileName);
                File logFileUploaded = new File(this.targetDirectory, logFileName);
                if (toBackup.exists()) {
                    toBackup.delete();
                }
                if (logFile.exists()) {
                    logFileUploaded.delete();
                    logFile.renameTo(logFileUploaded);
                }
                log.info((Object)("Upload succesful, saving backup to " + toBackup.getAbsolutePath()));
                if (!toUpload.renameTo(toBackup)) {
                    FileOutputStream fout = new FileOutputStream(toBackup);
                    FileInputStream fin = new FileInputStream(toUpload);
                    Algorithms.streamCopy((InputStream)fin, (OutputStream)fout);
                    fin.close();
                    fout.close();
                    toUpload.delete();
                }
                toBackup.setLastModified(System.currentTimeMillis());
                if (srcFile.equals(zipFile)) {
                    srcFile.delete();
                }
            }
        }
        catch (IOException e) {
            log.error((Object)("Input/output exception uploading " + fileName), (Throwable)e);
        }
    }

    private static String extractDirectory(String[] args, int ind) throws IndexUploadException {
        if (args.length > ind) {
            if ("-h".equals(args[0])) {
                throw new IndexUploadException("Usage: IndexZipper [directory] (if not specified, the current one will be taken)");
            }
            return args[ind];
        }
        return ".";
    }

    public boolean uploadFileToServer(File toUpload, String summary, UploadCredentials credentials) throws IOException {
        double originalLength = (double)toUpload.length() / 1048576.0;
        MessageFormat dateFormat = new MessageFormat("{0,date,dd.MM.yyyy}", Locale.US);
        MessageFormat numberFormat = new MessageFormat("{0,number,##.#}", Locale.US);
        String size = numberFormat.format(new Object[]{originalLength});
        String date = dateFormat.format(new Object[]{new Date(toUpload.lastModified())});
        try {
            credentials.upload(this, toUpload, summary, size, date);
        }
        catch (IOException e) {
            log.error((Object)("Input/output exception uploading " + toUpload.getName()), (Throwable)e);
            return false;
        }
        catch (JSchException e) {
            log.error((Object)("Input/output exception uploading " + toUpload.getName()), (Throwable)e);
            return false;
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void uploadToSSH(File f, String description, String size, String date, UploadSSHCredentials cred) throws IOException, JSchException {
        String serverName;
        log.info((Object)("Uploading file " + f.getName() + " " + size + " MB " + date + " of " + description));
        JSch jSch = new JSch();
        boolean knownHosts = false;
        if (cred.knownHosts != null) {
            jSch.setKnownHosts(cred.knownHosts);
            knownHosts = true;
        }
        if (cred.privateKey != null) {
            jSch.addIdentity(cred.privateKey);
        }
        if ((serverName = cred.url).startsWith("ssh://")) {
            serverName = serverName.substring("ssh://".length());
        }
        Session session = jSch.getSession(cred.user, serverName);
        if (cred.password != null) {
            session.setPassword(cred.password);
        }
        if (!knownHosts) {
            Properties config = new Properties();
            config.put("StrictHostKeyChecking", "no");
            session.setConfig(config);
        }
        String rfile = cred.path + "/" + f.getName();
        String lfile = f.getAbsolutePath();
        session.connect();
        String command = "scp -p -t \"" + rfile + "\"";
        Channel channel = session.openChannel("exec");
        ((ChannelExec)channel).setCommand(command);
        OutputStream out = channel.getOutputStream();
        InputStream in = channel.getInputStream();
        channel.connect();
        if (IndexUploader.checkAck(in) != 0) {
            channel.disconnect();
            session.disconnect();
            return;
        }
        long filesize = new File(lfile).length();
        command = "C0644 " + filesize + " ";
        command = lfile.lastIndexOf(47) > 0 ? command + lfile.substring(lfile.lastIndexOf(47) + 1) : command + lfile;
        command = command + "\n";
        out.write(command.getBytes());
        out.flush();
        if (IndexUploader.checkAck(in) != 0) {
            channel.disconnect();
            session.disconnect();
            return;
        }
        byte[] buf = new byte[1024];
        try (FileInputStream fis = new FileInputStream(lfile);){
            int len;
            while ((len = fis.read(buf, 0, buf.length)) > 0) {
                out.write(buf, 0, len);
            }
        }
        fis = null;
        buf[0] = 0;
        out.write(buf, 0, 1);
        out.flush();
        if (IndexUploader.checkAck(in) != 0) {
            channel.disconnect();
            session.disconnect();
            return;
        }
        out.close();
        channel.disconnect();
        session.disconnect();
        log.info((Object)"Finish uploading file index");
    }

    static int checkAck(InputStream in) throws IOException {
        int b = in.read();
        if (b == 0) {
            return b;
        }
        if (b == -1) {
            return b;
        }
        if (b == 1 || b == 2) {
            int c;
            StringBuilder sb = new StringBuilder();
            do {
                c = in.read();
                sb.append((char)c);
            } while (c != 10);
            if (b == 1) {
                System.out.print(sb.toString());
            }
            if (b == 2) {
                System.out.print(sb.toString());
            }
        }
        return b;
    }

    public static class DummyCredentials
    extends UploadCredentials {
        @Override
        public void upload(IndexUploader uploader, File toUpload, String summary, String size, String date) throws IOException, JSchException {
            log.info((Object)("Uploading dummy file " + toUpload.getName() + " " + size + " MB " + date + " of "));
        }
    }

    public static abstract class UploadCredentials {
        String password;
        String user;
        String url;
        String path;

        protected void parseParameter(String param) throws IndexUploadException {
            if (param.startsWith("--url=")) {
                this.url = param.substring("--url=".length());
            } else if (param.startsWith("--password=")) {
                this.password = param.substring("--password=".length());
            } else if (param.startsWith("--user=")) {
                this.user = param.substring("--user=".length());
            } else if (param.startsWith("--path=")) {
                this.path = param.substring("--path=".length());
            }
        }

        public boolean checkIfUploadNeededByTimestamp(String filename, long time) {
            return true;
        }

        public void disconnect() {
        }

        public void connect() throws IndexUploadException {
        }

        public abstract void upload(IndexUploader var1, File var2, String var3, String var4, String var5) throws IOException, JSchException;
    }

    protected static class FileFilter {
        private List<Pattern> matchers = null;
        private List<Pattern> exculdeMatchers = null;

        protected FileFilter() {
        }

        public void parseCSV(String patterns) {
            String[] s = patterns.split(",");
            if (this.matchers == null) {
                this.matchers = new ArrayList<Pattern>();
            }
            for (String p : s) {
                this.matchers.add(Pattern.compile(p.trim()));
            }
        }

        public void parseExcludeCSV(String patterns) {
            String[] s = patterns.split(",");
            if (this.exculdeMatchers == null) {
                this.exculdeMatchers = new ArrayList<Pattern>();
            }
            for (String p : s) {
                this.exculdeMatchers.add(Pattern.compile(p.trim()));
            }
        }

        public void parseFile(String file) throws IOException {
            File f = new File(file);
            if (f.exists()) {
                this.readPatterns(f);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void readPatterns(File f) throws IOException {
            BufferedReader reader = new BufferedReader(new FileReader(f));
            try {
                String line;
                if (this.matchers == null) {
                    this.matchers = new ArrayList<Pattern>();
                }
                while ((line = reader.readLine()) != null) {
                    this.matchers.add(Pattern.compile(line));
                }
            }
            finally {
                Algorithms.closeStream((Closeable)reader);
            }
        }

        public boolean patternMatches(String name) {
            return this.internalPatternMatches(name, this.matchers);
        }

        public boolean internalPatternMatches(String name, List<Pattern> matchers) {
            if (matchers == null) {
                return true;
            }
            for (Pattern p : matchers) {
                if (!p.matcher(name).matches()) continue;
                return true;
            }
            return false;
        }

        protected boolean fileCanBeUploaded(File f) {
            if (!f.isFile() || f.getName().endsWith(".gen.log")) {
                return false;
            }
            boolean matches = this.internalPatternMatches(f.getName(), this.matchers);
            if (matches && this.exculdeMatchers != null && this.internalPatternMatches(f.getName(), this.exculdeMatchers)) {
                return false;
            }
            return matches;
        }
    }

    public static class IndexUploadException
    extends Exception {
        private static final long serialVersionUID = 2343219168909577070L;

        public IndexUploadException(String message) {
            super(message);
        }

        public IndexUploadException(String message, Throwable e) {
            super(message, e);
        }
    }

    public static class UploadSSHCredentials
    extends UploadCredentials {
        String privateKey;
        String knownHosts;

        @Override
        protected void parseParameter(String param) throws IndexUploadException {
            super.parseParameter(param);
            if (param.startsWith("--privKey=")) {
                this.privateKey = param.substring("--privKey=".length());
            } else if (param.startsWith("--knownHosts=")) {
                this.knownHosts = param.substring("--knownHosts=".length());
            }
        }

        @Override
        public void upload(IndexUploader uploader, File toUpload, String summary, String size, String date) throws IOException, JSchException {
            uploader.uploadToSSH(toUpload, summary, size, date, this);
        }
    }

    public static class OneFileException
    extends Exception {
        private static final long serialVersionUID = 6463200194419498979L;

        public OneFileException(String message) {
            super(message);
        }

        public OneFileException(String message, Exception e) {
            super(message, e);
        }
    }
}

